package com.burhanstore.neonbubblequest.Utils

import android.os.CountDownTimer

object TimerManager {

    private var countDownTimer: CountDownTimer? = null
    private var millisLeft: Long = 0L
    private var totalMillis: Long = 0L
    private var isRunning = false

    interface TimerListener {
        fun onTick(formattedTime: String)
        fun onFinish()
    }

    private var listener: TimerListener? = null

    fun startTimerInSeconds(seconds: Long, timerListener: TimerListener) {
        startTimer(seconds * 1000, timerListener)
    }

    fun startTimerInMinutes(minutes: Long, timerListener: TimerListener) {
        startTimer(minutes * 60 * 1000, timerListener)
    }

    private fun startTimer(durationInMillis: Long, timerListener: TimerListener) {
        cancel()
        listener = timerListener
        millisLeft = durationInMillis
        totalMillis = millisLeft
        isRunning = true

        countDownTimer = object : CountDownTimer(millisLeft, 1000) {
            override fun onTick(millisUntilFinished: Long) {
                millisLeft = millisUntilFinished
                listener?.onTick(formatTime(millisUntilFinished))
            }

            override fun onFinish() {
                isRunning = false
                listener?.onFinish()
            }
        }.start()
    }

    fun cancel() {
        countDownTimer?.cancel()
        countDownTimer = null
        isRunning = false
        millisLeft = 0L
        listener = null
    }

    fun isTimerRunning(): Boolean = isRunning

    fun getRemainingMillis(): Long = millisLeft

    fun getElapsedSeconds(): Long {
        val played = (totalMillis - millisLeft) / 1000
        return if (played < 1) 1 else played
    }

    private fun formatTime(millis: Long): String {
        val totalSeconds = millis / 1000
        val minutes = totalSeconds / 60
        val seconds = totalSeconds % 60
        return if (minutes > 0)
            String.format("%02d:%02d", minutes, seconds)
        else
            String.format("%02d seconds", seconds)
    }
}
